<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class C1Support extends Model
{
    use SoftDeletes;

    /** @var string */
    protected $table = 'c1_supports';

    /** @var array */
    protected $fillable = [
        'supportable_id',
        'supportable_type',
        'type',
        'source',
        'value_amount',
        'details',
        'date_received',
    ];

    /** @var array */
    protected $casts = [
        'date_received' => 'date',
        'value_amount'  => 'decimal:2',
        'type'          => 'string',
        'source'        => 'string',
    ];

    // =========================================================
    // RELATIONSHIPS
    // =========================================================

    /**
     * The entity that received the support.
     * Can be: C1Group, C1Member, C1WeofBeneficiary (or future models)
     */
    public function supportable(): MorphTo
    {
        return $this->morphTo();
    }

    // =========================================================
    // SCOPES
    // =========================================================

    /** In-kind support only */
    public function scopeInKind($query)
    {
        return $query->where('type', 'in_kind');
    }

    /** Productive equipment only */
    public function scopeProductiveEquip($query)
    {
        return $query->where('type', 'productive_equip');
    }

    /** Grant tranches only */
    public function scopeGrantTranche($query)
    {
        return $query->where('type', 'grant_tranche');
    }

    /** Support from SSWSEEP */
    public function scopeFromSswseep($query)
    {
        return $query->where('source', 'sswseep');
    }

    /** Support from other sources */
    public function scopeFromOthers($query)
    {
        return $query->where('source', 'others');
    }

    /** Support given in a specific month/year */
    public function scopeInMonth($query, $year, $month = null)
    {
        $date = $month
            ? \Carbon\Carbon::createFromDate($year, $month, 1)
            : \Carbon\Carbon::createFromDate($year, 1, 1)->startOfYear();

        return $query->whereYear('date_received', $date->year)
                     ->whereMonth('date_received', $date->month);
    }

    // =========================================================
    // ACCESSORS
    // =========================================================

    /** Formatted amount for display */
    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->value_amount, 2);
    }

    /** Human-readable type */
    public function getTypeLabelAttribute(): string
    {
        return match ($this->type) {
            'in_kind'         => 'In-Kind Support',
            'productive_equip'=> 'Productive Equipment',
            'grant_tranche'   => 'Grant Tranche',
            default           => ucfirst(str_replace('_', ' ', $this->type)),
        };
    }

    /** Source label */
    public function getSourceLabelAttribute(): string
    {
        return $this->source === 'sswseep' ? 'SSWSEEP' : 'Other';
    }
}