<?php

namespace App\Http\Controllers\Component1;

use App\Http\Controllers\Controller;
use App\Models\C1Group;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class C1GroupController extends Controller
{
    /**
     * Display a listing of the groups.
     */
    public function index(): View
    {
        $groups = C1Group::withCount('members')
                         ->orderByDesc('created_at')
                         ->paginate(15);

        return view('component1.groups.index', compact('groups'));
    }

    /**
     * Show the form for creating a new group.
     */
    public function create(): View
    {
        return view('component1.groups.create');
    }

    /**
     * Store a newly created group.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name'                  => 'required|string|max:191',
            'nature'                => 'required|in:new,progressing,matured',
            'state'                 => 'required|string|max:100',
            'county'                => 'required|string|max:100',
            'payam'                 => 'required|string|max:100',
            'establishing_org_name' => 'nullable|string|max:191',
            'establishment_type'    => 'required|in:self,cso,ngo,government',
            'date_formation'        => 'required|date',
            'vsla_registration_date'=> 'nullable|date|after_or_equal:date_formation',
            'leadership'            => 'required|in:adolescent_girl,woman',
            'month'                 => 'required|date_format:Y-m',
            'average_participation' => 'nullable|numeric|min:0|max:100',
            'received_certificate'   => 'required|in:0,1',
        ]);

        C1Group::create($validated);

        return redirect()
            ->route('component1.groups.index')
            ->with('success', 'VSLA/Community Group created successfully.');
    }

    /**
     * Display the specified group.
     */
    public function show(C1Group $group): View
    {
        $group->load(['members' => fn($q) => $q->active()->limit(10)]);
        return view('component1.groups.show', compact('group'));
    }

    /**
     * Show the form for editing the group.
     */
    public function edit(C1Group $group): View
    {
        return view('component1.groups.edit', compact('group'));
    }

    /**
     * Update the specified group.
     */
    public function update(Request $request, C1Group $group): RedirectResponse
    {
        $validated = $request->validate([
            'name'                  => 'required|string|max:191',
            'nature'                => 'required|in:new,progressing,matured',
            'state'                 => 'required|string|max:100',
            'county'                => 'required|string|max:100',
            'payam'                 => 'required|string|max:100',
            'establishing_org_name' => 'nullable|string|max:191',
            'establishment_type'    => 'required|in:self,cso,ngo,government',
            'date_formation'        => 'required|date',
            'vsla_registration_date'=> 'nullable|date|after_or_equal:date_formation',
            'leadership'            => 'required|in:adolescent_girl,woman',
            'month'                 => 'required|date_format:Y-m',
            'average_participation' => 'nullable|numeric|min:0|max:100',
            'received_certificate'   => 'required|in:0,1',
        ]);

        $group->update($validated);

        return redirect()
            ->route('component1.groups.index')
            ->with('success', 'Group updated successfully.');
    }

    /**
     * Remove the specified group (soft delete).
     */
    public function destroy(C1Group $group): RedirectResponse
    {
        $group->delete();

        return redirect()
            ->route('component1.groups.index')
            ->with('success', 'Group deleted successfully.');
    }
}