<?php

namespace App\Http\Controllers\Component1;

use App\Http\Controllers\Controller;
use App\Models\C1Group;
use App\Models\C1WeofBeneficiary;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class C1WeofBeneficiaryController extends Controller
{
    public function index(Request $request): View
    {
        $query = C1WeofBeneficiary::with('group');

        // Filter by group
        if ($request->filled('group_id')) {
            $query->where('group_id', $request->group_id);
        }

        // Search
        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('business_name', 'like', '%' . $request->search . '%');
            });
        }

        $beneficiaries = $query->orderByDesc('created_at')->paginate(15);
        $groups = C1Group::orderBy('name')->pluck('name', 'id');

        return view('component1.weof.index', compact('beneficiaries', 'groups'));
    }

    public function create(Request $request): View
    {
        $groups = C1Group::orderBy('name')->pluck('name', 'id');
        $selectedGroup = $request->query('group_id');

        return view('component1.weof.create', compact('groups', 'selectedGroup'));
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'group_id' => 'nullable|exists:c1_groups,id',
            'woef_id' => 'required|string|max:50|unique:c1_weof_beneficiaries,woef_id',
            'name' => 'required|string|max:191',
            'age' => 'required|integer|min:18|max:100',
            'marital_status' => 'required|in:married,not_married,divorced,widowed',
            'resident_status' => 'required|in:resident,returnee,idp',
            'county' => 'required|string|max:100',
            'payam' => 'required|string|max:100',
            'education' => 'required|in:none,primary,secondary,tertiary,vocational',
            'phone_no' => 'nullable|string|max:20',
            'business_name' => 'required|string|max:191',
            'business_category' => 'required|string|max:100',
            'business_type' => 'required|in:existing,new',
            'establishment_year' => 'required|integer|min:1900|max:' . date('Y'),
            'registration_status' => 'required|boolean',
            'target_customers' => 'required|string|max:191',
            'monthly_revenue' => 'required|numeric|min:0',
            'financing_sources' => 'required|array|min:1',
            'financing_sources.*' => 'in:equity,loan,debt,grant',
            'ever_received_grant' => 'required|boolean',
            'ever_received_loan' => 'required|boolean',
            'employees_before_male' => 'required|integer|min:0',
            'employees_before_female' => 'required|integer|min:0',
            'received_training' => 'required|boolean',
            'training_type' => 'required_if:received_training,1|array|min:1',
            'training_type.*' => 'in:functional_literacy,vsla_ops,business_dev,digital_literacy,specialized',
            'training_source' => 'nullable|string|max:100',
            'participation' => 'required|in:successful,0.5,0.8,1_completed',
            'pre_test_score' => 'nullable|numeric|min:0|max:100',
            'post_test_score' => 'nullable|numeric|min:0|max:100',
            'pitch_outcome' => 'nullable|in:won,lost',
            'onsite_validation' => 'required|boolean',
            'accelerator_participation' => 'required|boolean',
            'accelerator_graduation' => 'required_if:accelerator_participation,1|boolean',
            'grant_1st_tranche' => 'nullable|numeric|min:0',
            'grant_2nd_tranche' => 'nullable|numeric|min:0',
            'business_plan_alignment' => 'nullable|numeric|min:0|max:100',
            'employees_now_male' => 'required|integer|min:0',
            'employees_now_female' => 'required|integer|min:0',
            'profitability' => 'nullable|numeric',
            'sales' => 'nullable|numeric|min:0',
            'additional_financing_equity' => 'nullable|numeric|min:0',
            'additional_financing_loan' => 'nullable|numeric|min:0',
            'additional_financing_debt' => 'nullable|numeric|min:0',
            'additional_financing_grant' => 'nullable|numeric|min:0',
            'participation_woef_connect' => 'required|boolean',
        ]);

        C1WeofBeneficiary::create($validated);

        return redirect()
            ->route('component1.weof.index', ['group_id' => $request->group_id])
            ->with('success', 'WEOF Beneficiary added successfully.');
    }

    public function show(C1WeofBeneficiary $weof): View
    {
        $weof->load('group');
        return view('component1.weof.show', compact('weof'));
    }

    public function edit(C1WeofBeneficiary $weof): View
    {
        $groups = C1Group::orderBy('name')->pluck('name', 'id');
        return view('component1.weof.edit', compact('weof', 'groups'));
    }

    public function update(Request $request, C1WeofBeneficiary $weof): RedirectResponse
    {
        $validated = $request->validate([
            'group_id' => 'nullable|exists:c1_groups,id',
            'woef_id' => 'required|string|max:50|unique:c1_weof_beneficiaries,woef_id,' . $weof->id,
            'name' => 'required|string|max:191',
            'age' => 'required|integer|min:18|max:100',
            'marital_status' => 'required|in:married,not_married,divorced,widowed',
            'resident_status' => 'required|in:resident,returnee,idp',
            'county' => 'required|string|max:100',
            'payam' => 'required|string|max:100',
            'education' => 'required|in:none,primary,secondary,tertiary,vocational',
            'phone_no' => 'nullable|string|max:20',
            'business_name' => 'required|string|max:191',
            'business_category' => 'required|string|max:100',
            'business_type' => 'required|in:existing,new',
            'establishment_year' => 'required|integer|min:1900|max:' . date('Y'),
            'registration_status' => 'required|boolean',
            'target_customers' => 'required|string|max:191',
            'monthly_revenue' => 'required|numeric|min:0',
            'financing_sources' => 'required|array|min:1',
            'financing_sources.*' => 'in:equity,loan,debt,grant',
            'ever_received_grant' => 'required|boolean',
            'ever_received_loan' => 'required|boolean',
            'employees_before_male' => 'required|integer|min:0',
            'employees_before_female' => 'required|integer|min:0',
            'received_training' => 'required|boolean',
            'training_type' => 'required_if:received_training,1|array|min:1',
            'training_type.*' => 'in:functional_literacy,vsla_ops,business_dev,digital_literacy,specialized',
            'training_source' => 'nullable|string|max:100',
            'participation' => 'required|in:successful,0.5,0.8,1_completed',
            'pre_test_score' => 'nullable|numeric|min:0|max:100',
            'post_test_score' => 'nullable|numeric|min:0|max:100',
            'pitch_outcome' => 'nullable|in:won,lost',
            'onsite_validation' => 'required|boolean',
            'accelerator_participation' => 'required|boolean',
            'accelerator_graduation' => 'required_if:accelerator_participation,1|boolean',
            'grant_1st_tranche' => 'nullable|numeric|min:0',
            'grant_2nd_tranche' => 'nullable|numeric|min:0',
            'business_plan_alignment' => 'nullable|numeric|min:0|max:100',
            'employees_now_male' => 'required|integer|min:0',
            'employees_now_female' => 'required|integer|min:0',
            'profitability' => 'nullable|numeric',
            'sales' => 'nullable|numeric|min:0',
            'additional_financing_equity' => 'nullable|numeric|min:0',
            'additional_financing_loan' => 'nullable|numeric|min:0',
            'additional_financing_debt' => 'nullable|numeric|min:0',
            'additional_financing_grant' => 'nullable|numeric|min:0',
            'participation_woef_connect' => 'required|boolean',
        ]);

        $weof->update($validated);

        return redirect()
            ->route('component1.weof.index', ['group_id' => $weof->group_id])
            ->with('success', 'WEOF Beneficiary updated.');
    }

    public function destroy(C1WeofBeneficiary $weof): RedirectResponse
    {
        $groupId = $weof->group_id;
        $weof->delete();

        return redirect()
            ->route('component1.weof.index', ['group_id' => $groupId])
            ->with('success', 'WEOF Beneficiary deleted.');
    }
}