<?php

namespace App\Http\Controllers\Component4;

use App\Http\Controllers\Controller;
use App\Models\C4Participant;
use App\Models\C4Indicator;
use Illuminate\Http\Request;
use App\Exports\C4ParticipantsExport;
use Maatwebsite\Excel\Facades\Excel;
use PDF; // We'll add this later for reports

class C4ParticipantController extends Controller
{
    // 1. Dashboard + Index
    public function index()
    {
        $participants = C4Participant::latest()->paginate(15);

        // Quick Stats (Same style as Component 1)
        $stats = [
            'total_participants' => C4Participant::count(),
            'total_female'       => C4Participant::where('gender', 'female')->count(),
            'total_male'         => C4Participant::where('gender', 'male')->count(),
            'trained_gbv'        => C4Participant::where('trained_gbv_programming', true)->count(),
            'trained_gender_eq'  => C4Participant::where('trained_gender_equality', true)->count(),
            'completed_all'      => C4Participant::where([
                ['trained_project_management', true],
                ['trained_grm', true],
                ['trained_monitoring_evaluation', true],
                ['trained_gbv_programming', true],
                ['trained_gender_equality', true],
                ['trained_resource_mobilization', true],
                ['trained_leadership_management', true],
            ])->count(),
        ];

        return view('component4.index', compact('participants', 'stats'));
    }

    // 2. Show registration form
    public function create()
    {
        return view('component4.create');
    }

    // 3. Store new participant
    public function store(Request $request)
    {
        $validated = $request->validate([
            'serial_no'          => 'nullable|integer',
            'name'               => 'required|string|max:191',
            'organization'       => 'nullable|string|max:191',
            'designation'        => 'nullable|string|max:191',
            'gender'             => 'required|in:female,male,other',
            'state_province'     => 'required|string|max:191',
            'start_date'         => 'nullable|date',
            'completion_date'    => 'nullable|date',
            'comment'            => 'nullable|string',

            // Training checkboxes
            'trained_project_management'     => 'sometimes|boolean',
            'trained_grm'                    => 'sometimes|boolean',
            'trained_monitoring_evaluation'  => 'sometimes|boolean',
            'trained_gbv_programming'        => 'sometimes|boolean',
            'trained_gender_equality'        => 'sometimes|boolean',
            'trained_resource_mobilization'  => 'sometimes|boolean',
            'trained_leadership_management' => 'sometimes|boolean',
        ]);

        // Laravel auto-converts checkboxes to boolean when using `sometimes|boolean`
        C4Participant::create($validated);

        return redirect()->route('component4.index')
            ->with('success', 'Participant registered successfully!');
    }

    // 4. Edit form
    public function edit(C4Participant $participant)
    {
        return view('component4.edit', compact('participant'));
    }

    // 5. Update participant
    public function update(Request $request, C4Participant $participant)
    {
        $validated = $request->validate([
            'serial_no'          => 'nullable|integer',
            'name'               => 'required|string|max:191',
            'organization'       => 'nullable|string|max:191',
            'designation'        => 'nullable|string|max:191',
            'gender'             => 'required|in:female,male,other',
            'state_province'     => 'required|string|max:191',
            'start_date'         => 'nullable|date',
            'completion_date'    => 'nullable|date',
            'comment'            => 'nullable|string',

            'trained_project_management'     => 'sometimes|boolean',
            'trained_grm'                    => 'sometimes|boolean',
            'trained_monitoring_evaluation'  => 'sometimes|boolean',
            'trained_gbv_programming'        => 'sometimes|boolean',
            'trained_gender_equality'        => 'sometimes|boolean',
            'trained_resource_mobilization'  => 'sometimes|boolean',
            'trained_leadership_management' => 'sometimes|boolean',
        ]);

        $participant->update($validated);

        return redirect()->route('component4.index')
            ->with('success', 'Participant updated successfully!');
    }

    // 6. Delete participant
    public function destroy(C4Participant $participant)
    {
        $participant->delete();

        return back()->with('success', 'Participant deleted successfully!');
    }
    public function export()
    {
        $filename = 'Component4_Training_Participants_' . now()->format('Y-m-d') . '.xlsx';
        return Excel::download(new C4ParticipantsExport, $filename);
    }
}