<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class C1WeofBeneficiary extends Model
{
    use SoftDeletes;

    /** @var string */
    protected $table = 'c1_weof_beneficiaries';

    /** @var array */
    protected $fillable = [
        'group_id',
        'woef_id',
        'name',
        'age',
        'marital_status',
        'resident_status',
        'county',
        'payam',
        'education',
        'phone_no',
        'business_name',
        'business_category',
        'business_type',
        'establishment_year',
        'registration_status',
        'target_customers',
        'monthly_revenue',
        'financing_sources',
        'ever_received_grant',
        'ever_received_loan',
        'employees_before_male',
        'employees_before_female',
        'received_training',
        'training_type',
        'training_source',
        'participation',
        'pre_test_score',
        'post_test_score',
        'pitch_outcome',
        'onsite_validation',
        'accelerator_participation',
        'accelerator_graduation',
        'grant_1st_tranche',
        'grant_2nd_tranche',
        'business_plan_alignment',
        'employees_now_male',
        'employees_now_female',
        'profitability',
        'sales',
        'additional_financing_equity',
        'additional_financing_loan',
        'additional_financing_debt',
        'additional_financing_grant',
        'participation_woef_connect',
    ];

    /** @var array */
    protected $casts = [
        'establishment_year' => 'integer',
        'financing_sources'   => 'array',
        'training_type'       => 'array',
        'marital_status'      => 'string',
        'participation'       => 'string',
        'registration_status' => 'boolean',
        'ever_received_grant' => 'boolean',
        'ever_received_loan'  => 'boolean',
        'received_training'   => 'boolean',
        'onsite_validation'   => 'boolean',
        'accelerator_participation' => 'boolean',
        'accelerator_graduation'    => 'boolean',
        'participation_woef_connect'=> 'boolean',

        // decimal fields
        'monthly_revenue'              => 'decimal:2',
        'pre_test_score'               => 'decimal:2',
        'post_test_score'              => 'decimal:2',
        'grant_1st_tranche'            => 'decimal:2',
        'grant_2nd_tranche'            => 'decimal:2',
        'business_plan_alignment'      => 'decimal:2',
        'profitability'                => 'decimal:2',
        'sales'                        => 'decimal:2',
        'additional_financing_equity'  => 'decimal:2',
        'additional_financing_loan'    => 'decimal:2',
        'additional_financing_debt'    => 'decimal:2',
        'additional_financing_grant'   => 'decimal:2',
    ];

    // =========================================================
    // RELATIONSHIPS
    // =========================================================

    /**
     * Optional link to the VSLA/community group the beneficiary belongs to.
     */
    public function group(): BelongsTo
    {
        return $this->belongsTo(C1Group::class, 'group_id');
    }

    /**
     * Polymorphic supports (in-kind, equipment, grant tranches, …)
     */
    public function supports(): MorphMany
    {
        return $this->morphMany(C1Support::class, 'supportable');
    }

    // =========================================================
    // SCOPES
    // =========================================================

    /** Active accelerator participants */
    public function scopeAcceleratorActive($query)
    {
        return $query->where('accelerator_participation', true);
    }

    /** Graduated from accelerator */
    public function scopeAcceleratorGraduated($query)
    {
        return $query->where('accelerator_graduation', true);
    }

    /** Received any grant tranche */
    public function scopeReceivedGrant($query)
    {
        return $query->where(function ($q) {
            $q->where('grant_1st_tranche', '>', 0)
              ->orWhere('grant_2nd_tranche', '>', 0);
        });
    }

    /** Completed full participation (1.0) */
    public function scopeFullyCompleted($query)
    {
        return $query->where('participation', '1_completed');
    }

    /** GBV-vulnerable (widowed, divorced, disability not directly stored – can be extended) */
    public function scopeGbvVulnerable($query)
    {
        return $query->whereIn('marital_status', ['widowed', 'divorced']);
    }

    // =========================================================
    // ACCESSORS & MUTATORS
    // =========================================================

    /** Total employees before training */
    public function getEmployeesBeforeTotalAttribute(): int
    {
        return $this->employees_before_male + $this->employees_before_female;
    }

    /** Total employees now */
    public function getEmployeesNowTotalAttribute(): int
    {
        return $this->employees_now_male + $this->employees_now_female;
    }

    /** Net employee growth */
    public function getEmployeeGrowthAttribute(): int
    {
        return $this->employeesNowTotal - $this->employeesBeforeTotal;
    }

    /** Total grant received */
    public function getTotalGrantAttribute(): float
    {
        return (float) $this->grant_1st_tranche + (float) $this->grant_2nd_tranche;
    }

    /** Total additional financing */
    public function getTotalAdditionalFinancingAttribute(): float
    {
        return (float) $this->additional_financing_equity
             + (float) $this->additional_financing_loan
             + (float) $this->additional_financing_debt
             + (float) $this->additional_financing_grant;
    }

    /** Formatted monthly revenue */
    public function getFormattedMonthlyRevenueAttribute(): string
    {
        return number_format($this->monthly_revenue, 2);
    }

    /** Participation label for UI */
    public function getParticipationLabelAttribute(): string
    {
        return match ($this->participation) {
            'successful'   => 'Successful',
            '0.5'          => '50% Alignment',
            '0.8'          => '80% Alignment',
            '1_completed'  => 'Fully Completed',
            default        => '–',
        };
    }

    /** Test score improvement */
    public function getTestScoreImprovementAttribute(): ?float
    {
        if (is_null($this->pre_test_score) || is_null($this->post_test_score)) {
            return null;
        }
        return round($this->post_test_score - $this->pre_test_score, 2);
    }
}