@php
    $layout = auth()->user()->role === 'super_admin' ? 'layouts.admin' : 'layouts.app';
@endphp

@extends($layout)

@section('title', 'Component 1 – Dashboard')

@push('styles')
    <style>
        /* DHIS2-inspired Color Palette */
        :root {
            --dhis2-blue-primary: #2c6693;
            --dhis2-blue-dark: #276696;
            --dhis2-blue-light: #4a90c2;
            --dhis2-teal: #00796b;
            --dhis2-grey-50: #f4f6f8;
            --dhis2-grey-100: #e8edf2;
            --dhis2-grey-200: #d5dde5;
            --dhis2-grey-300: #a0adb8;
            --dhis2-grey-600: #6c7a89;
            --dhis2-grey-900: #212934;
            --success-green: #4caf50;
            --warning-orange: #ff9800;
            --danger-red: #f44336;
            --info-cyan: #00bcd4;
        }

        body {
            background-color: var(--dhis2-grey-50);
            font-family: 'Roboto', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        }

        #top-padding {
            padding-top: 24px !important;
        }

        /* Dashboard Header */
        .dashboard-header {
            background: white;
            border-bottom: 3px solid var(--dhis2-blue-primary);
            padding: 1.5rem 0;
            margin-bottom: 1.5rem;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
        }

        .dashboard-title {
            color: var(--dhis2-grey-900);
            font-size: 1.75rem;
            font-weight: 500;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .dashboard-title::before {
            content: '';
            width: 4px;
            height: 32px;
            background: var(--dhis2-blue-primary);
            border-radius: 2px;
        }

        .breadcrumb {
            background: transparent;
            padding: 0.5rem 0 0 0;
            margin: 0;
            font-size: 0.875rem;
        }

        .breadcrumb-item {
            color: var(--dhis2-grey-600);
        }

        .breadcrumb-item.active {
            color: var(--dhis2-blue-primary);
        }

        /* Action Buttons - DHIS2 Style */
        .btn-dhis2-primary {
            background: var(--dhis2-blue-primary);
            color: white;
            border: none;
            padding: 0.5rem 1.25rem;
            border-radius: 4px;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.2s ease;
            box-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }

        .btn-dhis2-primary:hover {
            background: var(--dhis2-blue-dark);
            color: white;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.15);
            transform: translateY(-1px);
        }

        .btn-dhis2-secondary {
            background: white;
            color: var(--dhis2-blue-primary);
            border: 1px solid var(--dhis2-blue-primary);
            padding: 0.5rem 1.25rem;
            border-radius: 4px;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.2s ease;
        }

        .btn-dhis2-secondary:hover {
            background: var(--dhis2-blue-primary);
            color: white;
            border-color: var(--dhis2-blue-primary);
        }

        /* Stats Cards - DHIS2 Style */
        .stat-card {
            background: white;
            border-radius: 6px;
            padding: 1.5rem;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.08);
            border-left: 4px solid var(--dhis2-blue-primary);
            transition: all 0.2s ease;
            height: 100%;
        }

        .stat-card:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12);
            transform: translateY(-2px);
        }

        .stat-card.success {
            border-left-color: var(--success-green);
        }

        .stat-card.info {
            border-left-color: var(--info-cyan);
        }

        .stat-card.warning {
            border-left-color: var(--warning-orange);
        }

        .stat-card.primary {
            border-left-color: var(--dhis2-blue-primary);
        }

        .stat-label {
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: var(--dhis2-grey-600);
            margin-bottom: 0.5rem;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 500;
            color: var(--dhis2-grey-900);
            line-height: 1;
        }

        .stat-icon {
            font-size: 2.5rem;
            opacity: 0.15;
            position: absolute;
            right: 1.5rem;
            top: 50%;
            transform: translateY(-50%);
        }

        /* Chart Cards - DHIS2 Style */
        .chart-card {
            background: white;
            border-radius: 6px;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.08);
            overflow: hidden;
            margin-bottom: 1.5rem;
            transition: all 0.2s ease;
        }

        .chart-card:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12);
        }

        .chart-header {
            background: var(--dhis2-grey-50);
            border-bottom: 1px solid var(--dhis2-grey-200);
            padding: 1rem 1.25rem;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .chart-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: var(--dhis2-grey-900);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .chart-title i {
            color: var(--dhis2-blue-primary);
            font-size: 1rem;
        }

        .chart-body {
            padding: 1.5rem;
            background: white;
        }

        .chart-actions {
            display: flex;
            gap: 0.5rem;
        }

        .chart-action-btn {
            background: transparent;
            border: none;
            color: var(--dhis2-grey-600);
            padding: 0.25rem 0.5rem;
            cursor: pointer;
            border-radius: 3px;
            transition: all 0.2s ease;
            font-size: 0.875rem;
        }

        .chart-action-btn:hover {
            background: var(--dhis2-grey-100);
            color: var(--dhis2-blue-primary);
        }

        /* Filter Bar - DHIS2 Style */
        .filter-bar {
            background: white;
            border-radius: 6px;
            padding: 1rem 1.25rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.08);
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .filter-label {
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--dhis2-grey-600);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .filter-badge {
            background: var(--dhis2-blue-primary);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        /* Grid Layout */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 1.25rem;
            margin-bottom: 2rem;
        }

        .charts-grid {
            display: grid;
            grid-template-columns: repeat(12, 1fr);
            gap: 1.5rem;
        }

        .chart-col-6 {
            grid-column: span 6;
        }

        .chart-col-4 {
            grid-column: span 4;
        }

        .chart-col-8 {
            grid-column: span 8;
        }

        .chart-col-12 {
            grid-column: span 12;
        }

        /* Responsive */
        @media (max-width: 1200px) {

            .chart-col-6,
            .chart-col-4,
            .chart-col-8 {
                grid-column: span 12;
            }
        }

        @media (max-width: 768px) {
            .dashboard-header-content {
                flex-direction: column;
                gap: 1rem;
            }

            .dashboard-actions {
                width: 100%;
                justify-content: flex-start;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }
        }

        /* Loading State */
        .loading-skeleton {
            background: linear-gradient(90deg, var(--dhis2-grey-100) 25%, var(--dhis2-grey-200) 50%, var(--dhis2-grey-100) 75%);
            background-size: 200% 100%;
            animation: loading 1.5s ease-in-out infinite;
            border-radius: 4px;
            height: 300px;
        }

        @keyframes loading {
            0% {
                background-position: 200% 0;
            }

            100% {
                background-position: -200% 0;
            }
        }

        /* Negative Margin Utilities */
        .negative-mt-5 {
            margin-top: -50px !important;
        }
    </style>
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <style>
        .map-container {
            height: 300px;
            margin-bottom: 1.5rem;
        }
    </style>
@endpush

@section('content')
    <div class="container-fluid px-4 py-3">
        <!-- Dashboard Header -->
        <div class="dashboard-header">
            <div class="container-fluid">
                <div class="row align-items-center dashboard-header-content">
                    <div class="col-lg-6">
                        <h1 class="dashboard-title">
                            <i class="fas fa-tachometer-alt" style="color: var(--dhis2-blue-primary);"></i>
                            Component 1 Dashboard
                        </h1>
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb">
                                <li class="breadcrumb-item"><a href="{{ route('home') }}">Home</a></li>
                                <li class="breadcrumb-item active">Component 1</li>
                            </ol>
                        </nav>
                    </div>
                    <div class="col-lg-6">
                        <div class="d-flex justify-content-lg-end gap-2 dashboard-actions">
                            <a href="{{ route('component1.groups.index') }}" class="btn btn-dhis2-secondary">
                                <i class="fas fa-users me-1"></i> Manage Groups
                            </a>
                            <a href="{{ route('component1.members.index') }}" class="btn btn-dhis2-secondary">
                                <i class="fas fa-user-friends me-1"></i> Members
                            </a>
                            <a href="{{ route('component1.vsla.report') }}" class="btn btn-dhis2-secondary">
                                <i class="fas fa-file-excel"></i> VSLA Monthly Report
                            </a>
                            <!-- <a href="{{ route('component1.weof.index') }}" class="btn btn-dhis2-primary">
                                <i class="fas fa-briefcase me-1"></i> WEOF Beneficiaries
                            </a> -->
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Stats Cards Grid -->
        <div class="stats-grid">
            <div class="stat-card success">
                <div style="position: relative;">
                    <div class="stat-label">VSLA Groups</div>
                    <div class="stat-value">{{ number_format($stats['total_groups']) }}</div>
                    <i class="fas fa-users stat-icon" style="color: var(--success-green);"></i>
                </div>
            </div>

            <div class="stat-card primary">
                <div style="position: relative;">
                    <div class="stat-label">Total Members</div>
                    <div class="stat-value">{{ number_format($stats['total_members']) }}</div>
                    <i class="fas fa-user-friends stat-icon" style="color: var(--dhis2-blue-primary);"></i>
                </div>
            </div>

            {{-- <div class="stat-card warning">
            <div style="position: relative;">
                <div class="stat-label">GBV Vulnerable</div>
                <div class="stat-value">{{ number_format($gbvVulnerable) }}</div>
                <i class="fas fa-shield-alt stat-icon" style="color: var(--warning-orange);"></i>
            </div>
        </div> --}}

            <div class="stat-card success">
                <div style="position: relative;">
                    <div class="stat-label">Total Supports</div>
                    <div class="stat-value">{{ number_format($kpiStats['supports']) }}</div>
                    <i class="fas fa-hands-helping stat-icon" style="color: var(--success-green);"></i>
                </div>
            </div>
        </div>
        <hr />
        {{-- Present the excel formated data --}}

        <!-- VSLA Engagement Data Section (Table like Excel) -->
        <div class="container mb-5">
            <h3 class="mb-3">VSLA Engagement Data</h3>
            <div class="row">
                <div class="col-md-8">
                    <table class="table table-bordered bg-white shadow-sm rounded">
                        <thead class="bg-light">
                            <tr>
                                <th>VSLA Category</th>
                                <th>RF Target</th>
                                <th>Achieved</th>
                                <th>% Progress</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>Mature</td>
                                <td>240</td>
                                <td>{{ $groupsByNature->where('nature', 'matured')->first()->achieved ?? 0 }}</td>
                                <td>{{ round((($groupsByNature->where('nature', 'matured')->first()->achieved ?? 0) / 240) * 100, 2) }}%
                                </td>
                            </tr>
                            <tr>
                                <td>Progressive</td>
                                <td>160</td>
                                <td>{{ $groupsByNature->where('nature', 'progressive')->first()->achieved ?? 0 }}</td>
                                <td>{{ round((($groupsByNature->where('nature', 'progressive')->first()->achieved ?? 0) / 160) * 100, 2) }}%
                                </td>
                            </tr>
                            <tr>
                                <td>New</td>
                                <td>400</td>
                                <td>{{ $groupsByNature->where('nature', 'new')->first()->achieved ?? 0 }}</td>
                                <td>{{ round((($groupsByNature->where('nature', 'new')->first()->achieved ?? 0) / 400) * 100, 2) }}%
                                </td>
                            </tr>
                            <tr class="font-weight-bold">
                                <td>Total</td>
                                <td>800</td>
                                <td>{{ $groupsByNature->sum('achieved') }}</td>
                                <td>{{ round(($groupsByNature->sum('achieved') / 800) * 100, 2) }}%</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                <div class="col-md-4">
                    <canvas id="vslaCategoryChart" height="200"></canvas> <!-- Bar Chart -->
                </div>
            </div>
        </div>
        <hr />
        <!-- VSLA Membership Section -->
        <div class="container mb-5" id="top-padding">
            <h3 class="mb-3">VSLA Membership</h3>
            <div class="row">
                <!-- Membership Table (left side) -->
                <div class="col-md-8">
                    <div class="card shadow-sm">
                        <div class="card-body p-0">
                            <table class="table table-bordered mb-0">
                                <thead class="bg-light">
                                    <tr>
                                        <th class="text-center">VSLA Category</th>
                                        <th class="text-center">RF Target</th>
                                        <th class="text-center">Female</th>
                                        <th class="text-center">Male</th>
                                        <th class="text-center">Adolescent Girls</th>
                                        <th class="text-center">Total</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <!-- Mature Row -->
                                    <tr>
                                        <td>Mature</td>
                                        <td class="text-center">240</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['matured']['female'] ?? 0) }}</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['matured']['male'] ?? 0) }}</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['matured']['adol_girls'] ?? 0) }}</td>
                                        <td class="text-end fw-bold">
                                            {{ number_format($membershipByNature2['matured']['total'] ?? 0) }}</td>
                                    </tr>

                                    <!-- Progressive Row (now added) -->
                                    <tr>
                                        <td>Progressive</td>
                                        <td class="text-center">160</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['progressive']['female'] ?? 0) }}</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['progressive']['male'] ?? 0) }}</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['progressive']['adol_girls'] ?? 0) }}
                                        </td>
                                        <td class="text-end fw-bold">
                                            {{ number_format($membershipByNature2['progressive']['total'] ?? 0) }}</td>
                                    </tr>

                                    <!-- New Row -->
                                    <tr>
                                        <td>New</td>
                                        <td class="text-center">400</td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['new']['female'] ?? 0) }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2['new']['male'] ?? 0) }}
                                        </td>
                                        <td class="text-end">
                                            {{ number_format($membershipByNature2['new']['adol_girls'] ?? 0) }}</td>
                                        <td class="text-end fw-bold">
                                            {{ number_format($membershipByNature2['new']['total'] ?? 0) }}</td>
                                    </tr>

                                    <!-- Total Row -->
                                    <tr class="table-success fw-bold">
                                        <td>Total</td>
                                        <td class="text-center">800</td>
                                        <td class="text-end">{{ number_format($membershipByNature2->sum('female')) }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2->sum('male')) }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2->sum('adol_girls')) }}
                                        </td>
                                        <td class="text-end">{{ number_format($membershipByNature2->sum('total')) }}</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Doughnut Chart (right side) -->
                <div class="col-md-4 negative-mt-5">
                    <div class="card shadow-sm">
                        <div class="card-body">
                            <canvas id="membershipDoughnut" height="200"></canvas>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <hr />
    <!-- VSLA Support Data Header -->
    <div class="container mb-5">
        <h3 class="mb-3">VSLA Support Data</h3>
    </div>

    <!-- 1. VSLA Members Receiving Trainings Cumulative -->
    <div class="container mb-5">
        <h4 class="mb-3">VSLA Members Receiving Trainings Cumulative</h4>
        <div class="row">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-body p-0">
                        <table class="table table-bordered mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th rowspan="2" class="text-center align-middle">VSLA Category</th>
                                    <th rowspan="2" class="text-center align-middle">RF Target</th>
                                    <th colspan="4" class="text-center">Cumulative as of Dec 2025</th>
                                </tr>
                                <tr>
                                    <th class="text-center">Female</th>
                                    <th class="text-center">Male</th>
                                    <th class="text-center">Adolescent Girls</th>
                                    <th class="text-center">Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Mature</td>
                                    <td class="text-center">240</td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['matured']['female'] ?? 0) }}</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative['matured']['male'] ?? 0) }}
                                    </td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['matured']['adol_girls'] ?? 0) }}</td>
                                    <td class="text-end fw-bold">
                                        {{ number_format($trainingsCumulative['matured']['total'] ?? 0) }}</td>
                                </tr>
                                <tr>
                                    <td>Progressive</td>
                                    <td class="text-center">160</td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['progressive']['female'] ?? 0) }}</td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['progressive']['male'] ?? 0) }}</td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['progressive']['adol_girls'] ?? 0) }}</td>
                                    <td class="text-end fw-bold">
                                        {{ number_format($trainingsCumulative['progressive']['total'] ?? 0) }}</td>
                                </tr>
                                <tr>
                                    <td>New</td>
                                    <td class="text-center">400</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative['new']['female'] ?? 0) }}
                                    </td>
                                    <td class="text-end">{{ number_format($trainingsCumulative['new']['male'] ?? 0) }}
                                    </td>
                                    <td class="text-end">
                                        {{ number_format($trainingsCumulative['new']['adol_girls'] ?? 0) }}</td>
                                    <td class="text-end fw-bold">
                                        {{ number_format($trainingsCumulative['new']['total'] ?? 0) }}</td>
                                </tr>
                                <tr class="table-success fw-bold">
                                    <td>Total</td>
                                    <td class="text-center">800</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative->sum('female')) }}</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative->sum('male')) }}</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative->sum('adol_girls')) }}</td>
                                    <td class="text-end">{{ number_format($trainingsCumulative->sum('total')) }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <canvas id="trainingsCumulativeChart" height="300"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 2. VSLA Groups that Received Kits Cumulative -->
    <div class="container mb-5">
        <h4 class="mb-3">VSLA Groups that Received Kits Cumulative</h4>
        <div class="row">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-body p-0">
                        <table class="table table-bordered mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th rowspan="2" class="text-center align-middle">VSLA Category</th>
                                    <th colspan="4" class="text-center">Cumulative as of Dec 2025</th>
                                </tr>
                                <tr>
                                    <th class="text-center">Groups Received Kits</th>
                                    <th class="text-center">Female Led</th>
                                    <th class="text-center">Male Involved</th>
                                    <th class="text-center">Total Members Benefited</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Mature</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['matured']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['matured']['female_led'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['matured']['male'] ?? 0 }}</td>
                                    <td class="text-end fw-bold">{{ $kitsGroupsCumulative['matured']['members'] ?? 0 }}
                                    </td>
                                </tr>
                                <tr>
                                    <td>Progressive</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['progressive']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['progressive']['female_led'] ?? 0 }}
                                    </td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['progressive']['male'] ?? 0 }}</td>
                                    <td class="text-end fw-bold">
                                        {{ $kitsGroupsCumulative['progressive']['members'] ?? 0 }}</td>
                                </tr>
                                <tr>
                                    <td>New</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['new']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['new']['female_led'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative['new']['male'] ?? 0 }}</td>
                                    <td class="text-end fw-bold">{{ $kitsGroupsCumulative['new']['members'] ?? 0 }}</td>
                                </tr>
                                <tr class="table-success fw-bold">
                                    <td>Total</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative->sum('groups') }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative->sum('female_led') }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative->sum('male') }}</td>
                                    <td class="text-end">{{ $kitsGroupsCumulative->sum('members') }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <canvas id="kitsGroupsChart" height="300"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 3. VSLA Members that Received Kits Cumulative -->
    <div class="container mb-5">
        <h4 class="mb-3">VSLA Members that Received Kits Cumulative</h4>
        <div class="row">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-body p-0">
                        <table class="table table-bordered mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th rowspan="2" class="text-center align-middle">Category</th>
                                    <th colspan="2" class="text-center">Wau</th>
                                    <th colspan="2" class="text-center">Jur River</th>
                                    <th colspan="2" class="text-center">Total</th>
                                </tr>
                                <tr>
                                    <th class="text-center">Female</th>
                                    <th class="text-center">Male</th>
                                    <th class="text-center">Female</th>
                                    <th class="text-center">Male</th>
                                    <th class="text-center">Female</th>
                                    <th class="text-center">Male</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Adolescent Girls</td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adol_girls']['wau']['female'] ?? 0 }}
                                    </td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adol_girls']['wau']['male'] ?? 0 }}
                                    </td>
                                    <td class="text-end">
                                        {{ $kitsMembersCumulative['adol_girls']['jur_river']['female'] ?? 0 }}</td>
                                    <td class="text-end">
                                        {{ $kitsMembersCumulative['adol_girls']['jur_river']['male'] ?? 0 }}</td>
                                    <td class="text-end fw-bold">
                                        {{ ($kitsMembersCumulative['adol_girls']['wau']['female'] ?? 0) + ($kitsMembersCumulative['adol_girls']['jur_river']['female'] ?? 0) }}
                                    </td>
                                    <td class="text-end fw-bold">
                                        {{ ($kitsMembersCumulative['adol_girls']['wau']['male'] ?? 0) + ($kitsMembersCumulative['adol_girls']['jur_river']['male'] ?? 0) }}
                                    </td>
                                </tr>
                                <tr>
                                    <td>Adult</td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adult']['wau']['female'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adult']['wau']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adult']['jur_river']['female'] ?? 0 }}
                                    </td>
                                    <td class="text-end">{{ $kitsMembersCumulative['adult']['jur_river']['male'] ?? 0 }}
                                    </td>
                                    <td class="text-end fw-bold">
                                        {{ ($kitsMembersCumulative['adult']['wau']['female'] ?? 0) + ($kitsMembersCumulative['adult']['jur_river']['female'] ?? 0) }}
                                    </td>
                                    <td class="text-end fw-bold">
                                        {{ ($kitsMembersCumulative['adult']['wau']['male'] ?? 0) + ($kitsMembersCumulative['adult']['jur_river']['male'] ?? 0) }}
                                    </td>
                                </tr>
                                <tr class="table-success fw-bold">
                                    <td>Total</td>
                                    <td class="text-end">
                                        {{ ($kitsMembersCumulative['adol_girls']['wau']['female'] ?? 0) + ($kitsMembersCumulative['adult']['wau']['female'] ?? 0) }}
                                    </td>
                                    <td class="text-end">
                                        {{ ($kitsMembersCumulative['adol_girls']['wau']['male'] ?? 0) + ($kitsMembersCumulative['adult']['wau']['male'] ?? 0) }}
                                    </td>
                                    <td class="text-end">
                                        {{ ($kitsMembersCumulative['adol_girls']['jur_river']['female'] ?? 0) + ($kitsMembersCumulative['adult']['jur_river']['female'] ?? 0) }}
                                    </td>
                                    <td class="text-end">
                                        {{ ($kitsMembersCumulative['adol_girls']['jur_river']['male'] ?? 0) + ($kitsMembersCumulative['adult']['jur_river']['male'] ?? 0) }}
                                    </td>
                                    <td class="text-end">{{ $kitsMembersCumulative->sum('total_female') ?? 0 }}</td>
                                    <td class="text-end">{{ $kitsMembersCumulative->sum('total_male') ?? 0 }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <canvas id="kitsMembersChart" height="300"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 4. VSLA Groups who Received Productive Asset -->
    <div class="container mb-5">
        <h4 class="mb-3">VSLA Groups who Received Productive Asset</h4>
        <div class="row">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-body p-0">
                        <table class="table table-bordered mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th rowspan="2" class="text-center align-middle">VSLA Category</th>
                                    <th colspan="4" class="text-center">Cumulative as of Dec 2025</th>
                                </tr>
                                <tr>
                                    <th class="text-center">Groups</th>
                                    <th class="text-center">Female</th>
                                    <th class="text-center">Male</th>
                                    <th class="text-center">Adolescent Girls</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Mature</td>
                                    <td class="text-end">{{ $productiveAssetGroups['matured']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['matured']['female'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['matured']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['matured']['adol_girls'] ?? 0 }}</td>
                                </tr>
                                <tr>
                                    <td>Progressive</td>
                                    <td class="text-end">{{ $productiveAssetGroups['progressive']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['progressive']['female'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['progressive']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups['progressive']['adol_girls'] ?? 0 }}
                                    </td>
                                </tr>
                                <tr class="table-success fw-bold">
                                    <td>Total</td>
                                    <td class="text-end">{{ $productiveAssetGroups->sum('groups') }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups->sum('female') }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups->sum('male') }}</td>
                                    <td class="text-end">{{ $productiveAssetGroups->sum('adol_girls') }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <canvas id="productiveAssetChart" height="300"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- 5. VSLA Groups that Created and Operating New Business -->
    <div class="container mb-5">
        <h4 class="mb-3">VSLA Groups that Created and Operating New Business</h4>
        <div class="row">
            <div class="col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-body p-0">
                        <table class="table table-bordered mb-0">
                            <thead class="bg-light">
                                <tr>
                                    <th rowspan="2" class="text-center align-middle">VSLA Category</th>
                                    <th colspan="4" class="text-center">Cumulative as of Dec 2025</th>
                                </tr>
                                <tr>
                                    <th class="text-center">Groups</th>
                                    <th class="text-center">Female Led</th>
                                    <th class="text-center">Male Involved</th>
                                    <th class="text-center">Adolescent Girls</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>New</td>
                                    <td class="text-end">{{ $newBusinessGroups['new']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['new']['female_led'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['new']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['new']['adol_girls'] ?? 0 }}</td>
                                </tr>
                                <tr>
                                    <td>Progressive</td>
                                    <td class="text-end">{{ $newBusinessGroups['progressive']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['progressive']['female_led'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['progressive']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['progressive']['adol_girls'] ?? 0 }}</td>
                                </tr>
                                <tr>
                                    <td>Mature</td>
                                    <td class="text-end">{{ $newBusinessGroups['matured']['groups'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['matured']['female_led'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['matured']['male'] ?? 0 }}</td>
                                    <td class="text-end">{{ $newBusinessGroups['matured']['adol_girls'] ?? 0 }}</td>
                                </tr>
                                <tr class="table-success fw-bold">
                                    <td>Total</td>
                                    <td class="text-end">{{ $newBusinessGroups->sum('groups') }}</td>
                                    <td class="text-end">{{ $newBusinessGroups->sum('female_led') }}</td>
                                    <td class="text-end">{{ $newBusinessGroups->sum('male') }}</td>
                                    <td class="text-end">{{ $newBusinessGroups->sum('adol_girls') }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <canvas id="newBusinessChart" height="300"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <hr />
    <!-- Maps Section -->
    <div class="container mb-5">
        <div class="row">
            <div class="col-md-6">
                <h4>Group Locations Map</h4>
                <div id="groupMap" class="map-container"></div>
            </div>
            <div class="col-md-6">
                <h4>Member Locations Map</h4>
                <div id="memberMap" class="map-container"></div>
            </div>
        </div>
    </div>
    <hr />
    <!-- Quick Stats Cards -->
    <div class="container mb-5">
        <div class="row g-3">
            <div class="col-md-3">
                <div class="stat-card success">
                    <h5>Total Groups</h5>
                    <h2>{{ $totalGroups ?? 770 }} / 800</h2>
                    <small>96.25% Progress</small>
                </div>
            </div>
            <!-- More cards for Members, Trainings, etc. -->
        </div>
    </div>
    <hr />
    <!-- Charts Grid -->
    <div class="charts-grid">
        <!-- Groups by County -->
        <div class="chart-col-6">
            <div class="chart-card">
                <div class="chart-header">
                    <h6 class="chart-title">
                        <i class="fas fa-map-marked-alt"></i>
                        VSLA Groups by County (Top 10)
                    </h6>
                    <div class="chart-actions">
                        <button class="chart-action-btn" title="Download">
                            <i class="fas fa-download"></i>
                        </button>
                        <button class="chart-action-btn" title="Fullscreen">
                            <i class="fas fa-expand"></i>
                        </button>
                    </div>
                </div>
                <div class="chart-body">
                    <canvas id="groupsByCountyChart" height="280"></canvas>
                </div>
            </div>
        </div>

        <!-- Gender Distribution -->
        <div class="chart-col-6">
            <div class="chart-card">
                <div class="chart-header">
                    <h6 class="chart-title">
                        <i class="fas fa-venus-mars"></i>
                        Member Gender Distribution
                    </h6>
                    <div class="chart-actions">
                        <button class="chart-action-btn" title="Download">
                            <i class="fas fa-download"></i>
                        </button>
                        <button class="chart-action-btn" title="Fullscreen">
                            <i class="fas fa-expand"></i>
                        </button>
                    </div>
                </div>
                <div class="chart-body">
                    <canvas id="genderChart" height="280"></canvas>
                </div>
            </div>
        </div>

        <!-- Training Completion -->
        <div class="chart-col-4">
            <div class="chart-card">
                <div class="chart-header">
                    <h6 class="chart-title">
                        <i class="fas fa-graduation-cap"></i>
                        Core Training
                    </h6>
                    <div class="chart-actions">
                        <button class="chart-action-btn" title="Download">
                            <i class="fas fa-download"></i>
                        </button>
                        <button class="chart-action-btn" title="Fullscreen">
                            <i class="fas fa-expand"></i>
                        </button>
                    </div>
                </div>
                <div class="chart-body">
                    <canvas id="trainingChart" height="280"></canvas>
                </div>
            </div>
        </div>
        <div class="chart-col-4">
            <div class="chart-card">
                <div class="chart-header">
                    <h6 class="chart-title">
                        <i class="fas fa-graduation-cap"></i>
                        Groups By County
                    </h6>
                    <div class="chart-actions">
                        <button class="chart-action-btn" title="Download">
                            <i class="fas fa-download"></i>
                        </button>
                        <button class="chart-action-btn" title="Fullscreen">
                            <i class="fas fa-expand"></i>
                        </button>
                    </div>
                </div>
                <div class="chart-body">
                    <canvas id="groupsByCountyChart" height="280"></canvas>
                </div>
            </div>
        </div>
    </div>

    </div>

    @push('scripts')
        <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
        <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
        <script>
            // DHIS2 Color Palette
            const dhis2Colors = {
                primary: '#2c6693',
                secondary: '#00796b',
                success: '#4caf50',
                warning: '#ff9800',
                danger: '#f44336',
                info: '#00bcd4',
                grey: ['#6c7a89', '#a0adb8', '#d5dde5']
            };

            // Chart.js Global Defaults
            Chart.defaults.font.family = "'Roboto', sans-serif";
            Chart.defaults.font.size = 12;
            Chart.defaults.color = '#6c7a89';

            // Groups by County - Bar Chart
            const groupsByCounty = {!! json_encode($groupsByCounty->pluck('county')) !!};
            const groupsByCountyTotal = {!! json_encode($groupsByCounty->pluck('total')) !!};

            new Chart(document.getElementById('groupsByCountyChart'), {
                type: 'bar',
                data: {
                    labels: groupsByCounty,
                    datasets: [{
                        label: 'Number of Groups',
                        data: groupsByCountyTotal,
                        backgroundColor: dhis2Colors.primary,
                        borderRadius: 4,
                        borderSkipped: false,
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: 'rgba(33, 41, 52, 0.95)',
                            padding: 12,
                            borderRadius: 6,
                            titleFont: {
                                size: 13,
                                weight: '500'
                            },
                            bodyFont: {
                                size: 12
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: '#e8edf2',
                                drawBorder: false
                            },
                            ticks: {
                                padding: 8
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                padding: 8
                            }
                        }
                    }
                }
            });

            // Gender Distribution - Doughnut
            new Chart(document.getElementById('genderChart'), {
                type: 'doughnut',
                data: {
                    labels: ['Female Members', 'Male Members'],
                    datasets: [{
                        data: [{{ $genderData['female'] }}, {{ $genderData['male'] }}],
                        backgroundColor: [dhis2Colors.info, dhis2Colors.primary],
                        borderWidth: 0,
                        hoverOffset: 8
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 15,
                                usePointStyle: true,
                                font: {
                                    size: 12,
                                    weight: '500'
                                }
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(33, 41, 52, 0.95)',
                            padding: 12,
                            borderRadius: 6
                        }
                    },
                    cutout: '65%'
                }
            });

            // Training Completion - Bar
            new Chart(document.getElementById('trainingChart'), {
                type: 'bar',
                data: {
                    labels: {!! json_encode(array_keys($trainingCompletion)) !!},
                    datasets: [{
                        label: 'Members Completed',
                        data: {!! json_encode(array_values($trainingCompletion)) !!},
                        backgroundColor: dhis2Colors.warning,
                        borderRadius: 4,
                        borderSkipped: false,
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            backgroundColor: 'rgba(33, 41, 52, 0.95)',
                            padding: 12,
                            borderRadius: 6
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: '#e8edf2',
                                drawBorder: false
                            },
                            ticks: {
                                padding: 8
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            },
                            ticks: {
                                padding: 8,
                                maxRotation: 45,
                                minRotation: 45
                            }
                        }
                    }
                }
            });
            // New Bar Chart for VSLA Categories
            const vslaCtx = document.getElementById('vslaCategoryChart').getContext('2d');
            new Chart(vslaCtx, {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [{
                        label: 'RF Target',
                        data: [240, 160, 400],
                        backgroundColor: 'rgba(44, 102, 147, 0.6)'
                    }, {
                        label: 'Achieved',
                        data: [{{ $groupsByNature->where('nature', 'matured')->first()->achieved ?? 0 }},
                            {{ $groupsByNature->where('nature', 'progressive')->first()->achieved ?? 0 }},
                            {{ $groupsByNature->where('nature', 'new')->first()->achieved ?? 0 }}
                        ],
                        backgroundColor: 'rgba(0, 121, 107, 0.6)'
                    }]
                },
                options: {
                    /* Responsive options */ }
            });

            // Doughnut for Membership
            const membershipCtx = document.getElementById('membershipDoughnut').getContext('2d');
            new Chart(membershipCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Female', 'Male', 'Adolescent Girls'],
                    datasets: [{
                        data: [{{ $totalFemale ?? 17986 }}, {{ $totalMale ?? 2245 }},
                            {{ $totalAdolGirls ?? 3304 }}
                        ],
                        backgroundColor: ['#4caf50', '#ff9800', '#00bcd4']
                    }]
                },
                options: {
                    /* Cutout, legend, etc. */ }
            });
            
            new Chart(document.getElementById('trainingsCumulativeChart'), {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [
                        { label: 'Female', data: [{{ $trainingsCumulative['matured']['female'] ?? 0 }}, {{ $trainingsCumulative['progressive']['female'] ?? 0 }}, {{ $trainingsCumulative['new']['female'] ?? 0 }}], backgroundColor: '#4caf50' },
                        { label: 'Male', data: [{{ $trainingsCumulative['matured']['male'] ?? 0 }}, {{ $trainingsCumulative['progressive']['male'] ?? 0 }}, {{ $trainingsCumulative['new']['male'] ?? 0 }}], backgroundColor: '#ff9800' },
                        { label: 'Adolescent Girls', data: [{{ $trainingsCumulative['matured']['adol_girls'] ?? 0 }}, {{ $trainingsCumulative['progressive']['adol_girls'] ?? 0 }}, {{ $trainingsCumulative['new']['adol_girls'] ?? 0 }}], backgroundColor: '#00bcd4' }
                    ]
                },
                options: { responsive: true, plugins: { legend: { position: 'bottom' } } }
            });

            new Chart(document.getElementById('kitsGroupsChart'), {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [
                        { label: 'Groups Received Kits', data: [{{ $kitsGroupsCumulative['matured']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['progressive']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['new']['groups'] ?? 0 }}], backgroundColor: '#4caf50' },
                    ]
                },
            });
            new Chart(document.getElementById('kitsMembersChart'), {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [
                        { label: 'Members Received Kits', data: [{{ $kitsMembersCumulative['matured']['total'] ?? 0 }}, {{ $kitsMembersCumulative['progressive']['total'] ?? 0 }}, {{ $kitsMembersCumulative['new']['total'] ?? 0 }}], backgroundColor: '#4caf50' },
                    ]
                },
            });
            new Chart(document.getElementById('productiveAssetChart'), {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [
                        { label: 'Groups with Productive Assets', data: [{{ $productiveAssetGroups['matured']['groups'] ?? 0 }}, {{ $productiveAssetGroups['progressive']['groups'] ?? 0 }}, {{ $productiveAssetGroups['new']['groups'] ?? 0 }}], backgroundColor: '#4caf50' },
                    ]
                },
            });
            new Chart(document.getElementById('newBusinessChart'), {
                type: 'bar',
                data: {
                    labels: ['Mature', 'Progressive', 'New'],
                    datasets: [
                        { label: 'Groups with New Business', data: [{{ $newBusinessGroups['matured']['groups'] ?? 0 }}, {{ $newBusinessGroups['progressive']['groups'] ?? 0 }}, {{ $newBusinessGroups['new']['groups'] ?? 0 }}], backgroundColor: '#4caf50' },
                    ]
                },
            });
            // Maps
            const groupMap = L.map('groupMap', { scrollWheelZoom: false }).setView([7.0, 30.0], 6); // South Sudan center
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(groupMap);
            // Add markers from controller data, e.g., @json($groupLocations) where $groupLocations = [['lat' => 7.7, 'lon' => 28.0, 'name' => 'Wau Group']]
            const groupLocations = {!! json_encode($groupLocations) !!};
            groupLocations.forEach(loc => L.marker([loc.lat, loc.lon]).addTo(groupMap).bindPopup(loc.name));

            const memberMap = L.map('memberMap', { scrollWheelZoom: false }).setView([7.0, 30.0], 6);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(memberMap);
            const memberLocations = {!! json_encode($memberLocations) !!}; // Similar aggregation
            memberLocations.forEach(loc => L.marker([loc.lat, loc.lon]).addTo(memberMap).bindPopup(loc.name));

            // Add other charts similarly (line for progress, pie for trainings, bar for supports)
        </script>
    @endpush
@endsection
