@php
    $layout = auth()->user()->role === 'super_admin' ? 'layouts.admin' : 'layouts.app';
@endphp

@extends($layout)

@section('title', 'Component 1 – Executive Dashboard')

@push('styles')
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <style>
        /* --- PowerBI Professional Palette --- */
        :root {
            --bi-bg: #f0f2f5;
            --bi-card-bg: #ffffff;
            --bi-text-main: #1a1d21;
            --bi-text-muted: #64748b;
            --bi-border: #e2e8f0;
            --dhis2-blue: #2c6693;
            --dhis2-teal: #00796b;
            --dhis2-green: #22c55e;
            --dhis2-orange: #f59e0b;
        }

        body { background-color: var(--bi-bg); font-family: 'Inter', system-ui, sans-serif; color: var(--bi-text-main); }

        .bi-container { padding: 1.5rem; max-width: 100%; margin: 0 auto; }

        /* --- KPI Cards --- */
        .kpi-card {
            background: white;
            border-radius: 8px;
            padding: 1.25rem;
            border: 1px solid var(--bi-border);
            display: flex;
            flex-direction: column;
            position: relative;
            overflow: hidden;
            height: 110px;
        }
        .kpi-label { font-size: 0.75rem; font-weight: 600; color: var(--bi-text-muted); text-transform: uppercase; letter-spacing: 0.05em; }
        .kpi-value { font-size: 1.75rem; font-weight: 800; color: var(--bi-text-main); margin-top: 4px; }
        .kpi-icon { position: absolute; right: 1rem; top: 1rem; font-size: 1.5rem; color: var(--bi-border); }
        .kpi-border { position: absolute; left: 0; top: 0; bottom: 0; width: 4px; }

        /* --- Dashboard Cards --- */
        .bi-card {
            background: var(--bi-card-bg);
            border-radius: 8px;
            border: 1px solid var(--bi-border);
            box-shadow: 0 1px 2px rgba(0,0,0,0.05);
            display: flex;
            flex-direction: column;
            margin-bottom: 1.5rem;
        }
        .bi-card-header {
            padding: 1rem;
            border-bottom: 1px solid var(--bi-border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .bi-card-title { font-size: 0.875rem; font-weight: 700; color: var(--bi-text-main); text-transform: uppercase; margin: 0; }
        .bi-card-body { padding: 1rem; flex: 1; position: relative; }

        /* --- Tables --- */
        .table-bi { width: 100%; font-size: 0.85rem; margin-bottom: 0; }
        .table-bi th { background: #f8fafc; color: var(--bi-text-muted); font-weight: 600; text-transform: uppercase; font-size: 0.7rem; padding: 10px; border-bottom: 1px solid var(--bi-border); }
        .table-bi td { padding: 10px; border-bottom: 1px solid #f1f5f9; vertical-align: middle; }

        /* --- Tabs --- */
        .nav-tabs-bi { border-bottom: 1px solid var(--bi-border); gap: 1.5rem; padding: 0 1rem; }
        .nav-tabs-bi .nav-link { border: none; border-bottom: 3px solid transparent; padding: 0.75rem 0; font-weight: 600; color: var(--bi-text-muted); font-size: 0.85rem; }
        .nav-tabs-bi .nav-link.active { color: var(--dhis2-blue); border-bottom-color: var(--dhis2-blue); background: transparent; }

        .chart-container { position: relative; height: 250px; width: 100%; }
        .map-container { height: 350px; width: 100%; border-radius: 4px; }
    </style>
@endpush

@section('content')
<div class="bi-container">
    
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4 class="fw-bold mb-0" style="color: var(--dhis2-blue)">Executive Monitoring Dashboard</h4>
            <p class="text-muted small mb-0">Component 1: VSLA & Group Performance</p>
        </div>
        <div class="d-flex gap-2">
            <a href="{{ route('component1.vsla.report') }}" class="btn btn-sm btn-outline-primary"><i class="fas fa-download"></i> Export Data</a>
        </div>
    </div>

    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="kpi-card">
                <div class="kpi-border" style="background: var(--dhis2-blue)"></div>
                <div class="kpi-label">Active Groups</div>
                <div class="kpi-value">{{ number_format($stats['total_groups']) }}</div>
                <i class="fas fa-users kpi-icon"></i>
            </div>
        </div>
        <div class="col-md-3">
            <div class="kpi-card">
                <div class="kpi-border" style="background: var(--dhis2-teal)"></div>
                <div class="kpi-label">Total Members</div>
                <div class="kpi-value">{{ number_format($stats['total_members']) }}</div>
                <i class="fas fa-user-friends kpi-icon"></i>
            </div>
        </div>
        <div class="col-md-3">
            <div class="kpi-card">
                <div class="kpi-border" style="background: var(--dhis2-green)"></div>
                <div class="kpi-label">Supports Provided</div>
                <div class="kpi-value">{{ number_format($kpiStats['supports']) }}</div>
                <i class="fas fa-box-open kpi-icon"></i>
            </div>
        </div>
        <div class="col-md-3">
            <div class="kpi-card">
                <div class="kpi-border" style="background: var(--dhis2-orange)"></div>
                <div class="kpi-label">Target Progress</div>
                <div class="kpi-value">{{ round(($stats['total_groups'] / 800) * 100) }}%</div>
                <i class="fas fa-bullseye kpi-icon"></i>
            </div>
        </div>
    </div>

    <div class="row g-3">
        <div class="col-lg-8">
            
            <div class="bi-card">
                <div class="bi-card-header">
                    <h6 class="bi-card-title">Engagement Performance</h6>
                </div>
                <div class="bi-card-body">
                    <div class="row">
                        <div class="col-md-7">
                            <table class="table-bi">
                                <thead>
                                    <tr>
                                        <th>Group Nature</th>
                                        <th class="text-center">Target</th>
                                        <th class="text-center">Achieved</th>
                                        <th class="text-end">Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach(['matured' => 240, 'progressive' => 160, 'new' => 400] as $nature => $target)
                                    @php 
                                        $val = $groupsByNature->where('nature', $nature)->first()->achieved ?? 0; 
                                        $pct = ($val / $target) * 100;
                                    @endphp
                                    <tr>
                                        <td class="text-capitalize fw-bold">{{ $nature }}</td>
                                        <td class="text-center text-muted">{{ $target }}</td>
                                        <td class="text-center fw-bold">{{ $val }}</td>
                                        <td class="text-end">
                                            <span class="badge {{ $pct >= 80 ? 'bg-success' : 'bg-warning' }}">{{ round($pct) }}%</span>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <div class="col-md-5">
                            <div class="chart-container">
                                <canvas id="natureChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="bi-card">
                <div class="bi-card-header p-0">
                    <ul class="nav nav-tabs-bi" id="detailTabs" role="tablist">
                        <li class="nav-item"><a class="nav-link active" data-bs-toggle="tab" href="#tab-membership">Membership</a></li>
                        <li class="nav-item"><a class="nav-link" data-bs-toggle="tab" href="#tab-kits">Kits & Assets</a></li>
                    </ul>
                </div>
                <div class="bi-card-body">
                    <div class="tab-content">
                        <div class="tab-pane fade show active" id="tab-membership">
                            <table class="table-bi">
                                <thead>
                                    <tr>
                                        <th>Category</th>
                                        <th class="text-end">Female</th>
                                        <th class="text-end">Male</th>
                                        <th class="text-end">Adol. Girls</th>
                                        <th class="text-end">Total</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach(['matured', 'progressive', 'new'] as $nature)
                                    <tr>
                                        <td class="text-capitalize">{{ $nature }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2[$nature]['female'] ?? 0) }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2[$nature]['male'] ?? 0) }}</td>
                                        <td class="text-end">{{ number_format($membershipByNature2[$nature]['adol_girls'] ?? 0) }}</td>
                                        <td class="text-end fw-bold">{{ number_format($membershipByNature2[$nature]['total'] ?? 0) }}</td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <div class="tab-pane fade" id="tab-kits">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6 class="small fw-bold mb-3">Group Kits Received</h6>
                                    <div class="chart-container" style="height: 150px">
                                        <canvas id="kitsChart"></canvas>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <h6 class="small fw-bold mb-3">Productive Assets</h6>
                                    <div class="chart-container" style="height: 150px">
                                        <canvas id="assetsChart"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <div class="bi-card">
                <div class="bi-card-header">
                    <h6 class="bi-card-title">Geographic Spread</h6>
                </div>
                <div class="bi-card-body p-0">
                    <div id="groupMap" class="map-container"></div>
                </div>
            </div>

            <div class="bi-card">
                <div class="bi-card-header">
                    <h6 class="bi-card-title">Groups by County</h6>
                </div>
                <div class="bi-card-body">
                    <div class="chart-container" style="height: 300px">
                        <canvas id="countyChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    // Common Chart Styles
    Chart.defaults.font.family = "'Inter', sans-serif";
    Chart.defaults.color = '#64748b';

    // Nature Chart
    new Chart(document.getElementById('natureChart'), {
        type: 'doughnut',
        data: {
            labels: ['Matured', 'Progressive', 'New'],
            datasets: [{
                data: [
                    {{ $groupsByNature->where('nature', 'matured')->first()->achieved ?? 0 }},
                    {{ $groupsByNature->where('nature', 'progressive')->first()->achieved ?? 0 }},
                    {{ $groupsByNature->where('nature', 'new')->first()->achieved ?? 0 }}
                ],
                backgroundColor: ['#2c6693', '#00796b', '#f59e0b'],
                borderWidth: 0
            }]
        },
        options: { cutout: '70%', plugins: { legend: { position: 'bottom' } } }
    });

    // County Chart (Unique ID: countyChart)
    new Chart(document.getElementById('countyChart'), {
        type: 'bar',
        data: {
            labels: {!! json_encode($groupsByCounty->pluck('county')) !!},
            datasets: [{
                label: 'Groups',
                data: {!! json_encode($groupsByCounty->pluck('total')) !!},
                backgroundColor: '#2c6693',
                borderRadius: 4
            }]
        },
        options: { indexAxis: 'y', plugins: { legend: { display: false } }, maintainAspectRatio: false }
    });

    // Kits & Assets Charts
    new Chart(document.getElementById('kitsChart'), {
        type: 'bar',
        data: {
            labels: ['Matured', 'Progressive', 'New'],
            datasets: [{
                data: [{{ $kitsGroupsCumulative['matured']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['progressive']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['new']['groups'] ?? 0 }}],
                backgroundColor: '#00796b'
            }]
        },
        options: { plugins: { legend: { display: false } }, maintainAspectRatio: false }
    });

    new Chart(document.getElementById('assetsChart'), {
        type: 'bar',
        data: {
            labels: ['Matured', 'Progressive', 'New'],
            datasets: [{
                data: [{{ $productiveAssetGroups['matured']['groups'] ?? 0 }}, {{ $productiveAssetGroups['progressive']['groups'] ?? 0 }}, {{ $productiveAssetGroups['new']['groups'] ?? 0 }}],
                backgroundColor: '#2c6693'
            }]
        },
        options: { plugins: { legend: { display: false } }, maintainAspectRatio: false }
    });

    // Map Initialization
    const map = L.map('groupMap', { zoomControl: false }).setView([7.0, 30.0], 6);
    L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png').addTo(map);
    const locations = {!! json_encode($groupLocations) !!};
    locations.forEach(loc => {
        L.circleMarker([loc.lat, loc.lon], { radius: 5, color: '#2c6693', fillOpacity: 0.8 }).addTo(map).bindPopup(loc.name);
    });

    // Fix map rendering in tabs
    document.querySelectorAll('a[data-bs-toggle="tab"]').forEach(tab => {
        tab.addEventListener('shown.bs.tab', () => { window.dispatchEvent(new Event('resize')); });
    });
</script>
@endpush